"""
plot_results.py
~~~~~~~~~~~~~~~~

Placeholder module for plotting Wilson loop data.  The original Volume 4
pipeline produced matplotlib figures summarising the area and perimeter
scaling of the Wilson loops.  In this simplified implementation the
plotting step is optional.  If matplotlib is available the plots will be
generated; otherwise a friendly message is printed and the pipeline
continues without visualisations.
"""

from __future__ import annotations

import os
import yaml
import numpy as np
import pandas as pd


def main(config_path: str = 'config.yaml') -> None:
    # Attempt to import matplotlib lazily
    try:
        import matplotlib.pyplot as plt
    except ImportError:
        print('Matplotlib is not available – skipping plotting step.')
        return None
    cfg_file = config_path if os.path.isabs(config_path) else os.path.abspath(config_path)
    if not os.path.exists(cfg_file):
        raise FileNotFoundError(f"Cannot find configuration file: {config_path}")
    with open(cfg_file) as f:
        cfg = yaml.safe_load(f)
    base_dir = os.path.dirname(cfg_file)
    results_dir_cfg = cfg.get('results_dir', 'results')
    results_dir = results_dir_cfg if os.path.isabs(results_dir_cfg) else os.path.join(base_dir, results_dir_cfg)
    gauge_groups = [g.upper() for g in cfg.get('gauge_groups', ['U1'])]
    for group in gauge_groups:
        results_path = os.path.join(results_dir, f'wilson_{group}.csv')
        if not os.path.exists(results_path):
            continue
        df = pd.read_csv(results_path)
        # Plot real and imaginary parts versus loop area (L^2)
        sizes = df['size'].values
        areas = sizes ** 2
        plt.figure()
        plt.plot(areas, df['real'], 'o-', label='Re')
        plt.plot(areas, df['imag'], 's-', label='Im')
        plt.xlabel('Loop area (L^2)')
        plt.ylabel('⟨W(L)⟩')
        plt.title(f'Wilson loop expectation values ({group})')
        plt.legend()
        plt.grid(True)
        plot_path = os.path.join(results_dir, f'plot_wilson_{group}.png')
        plt.savefig(plot_path)
        plt.close()
    return None


if __name__ == '__main__':
    import sys
    cfg = sys.argv[1] if len(sys.argv) > 1 else 'config.yaml'
    main(cfg)